import os
import yaml
import numpy as np
import argparse
import matplotlib.pyplot as plt
import pandas as pd


def main(config_path: str):
    # Load config
    with open(config_path) as f:
        cfg = yaml.safe_load(f)

    # Resolve results_dir relative to the directory of the config file
    base_dir = os.path.dirname(os.path.abspath(config_path))
    results_dir_cfg = cfg.get('results_dir', 'results')
    if os.path.isabs(results_dir_cfg):
        results_dir = results_dir_cfg
    else:
        results_dir = os.path.join(base_dir, results_dir_cfg)
    gauge_groups = cfg.get('gauge_groups', ['U1'])

    for G in gauge_groups:
        csv_path = os.path.join(results_dir, f'wilson_{G}.csv')
        # Load Wilson loop averages using pandas.  The CSV files store the
        # real and imaginary parts of the average separately as ``real`` and
        # ``imag`` columns.  Using pandas avoids the need to parse complex
        # strings and ensures robust handling of the header.
        df = pd.read_csv(csv_path)
        # Extract loop sizes as a NumPy array
        sizes = df['size'].to_numpy()
        # Reconstruct complex averages from separate real/imag columns
        avg = df['real'].to_numpy() + 1j * df['imag'].to_numpy()
        # Use the magnitude of the complex average.  The absolute value is
        # necessary because the argument to the logarithm must be positive.
        log_avg = np.log(np.abs(avg))

        # Compute area and perimeter
        areas = sizes**2
        perimeters = 4 * sizes

        # Fit linear models: log_avg = m * x + c
        slope_area, intercept_area = np.polyfit(areas, log_avg, 1)
        slope_perim, intercept_perim = np.polyfit(perimeters, log_avg, 1)

        # Plot area-law
        plt.figure()
        plt.scatter(areas, log_avg)
        # Fit line values
        x_vals = np.linspace(min(areas), max(areas), 100)
        y_vals = slope_area * x_vals + intercept_area
        plt.plot(x_vals, y_vals)
        plt.xlabel('Area')
        plt.ylabel('log(|⟨W⟩|)')
        plt.title(f'Area‑law fit for {G}: slope={slope_area:.3f}')
        area_plot = os.path.join(results_dir, f'area_law_{G}.png')
        plt.savefig(area_plot)
        plt.close()
        print(f'Saved area‑law plot for {G} to {area_plot}')

        # Plot perimeter-law
        plt.figure()
        plt.scatter(perimeters, log_avg)
        x_vals = np.linspace(min(perimeters), max(perimeters), 100)
        y_vals = slope_perim * x_vals + intercept_perim
        plt.plot(x_vals, y_vals)
        plt.xlabel('Perimeter')
        plt.ylabel('log(|⟨W⟩|)')
        plt.title(f'Perimeter‑law fit for {G}: slope={slope_perim:.3f}')
        perim_plot = os.path.join(results_dir, f'perimeter_law_{G}.png')
        plt.savefig(perim_plot)
        plt.close()
        print(f'Saved perimeter‑law plot for {G} to {perim_plot}')


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Plot Wilson loop scaling results')
    parser.add_argument('--config', default='config.yaml', help='Path to config file')
    args = parser.parse_args()
    main(args.config)